/* PC parallel port DAC speaker driver. Accepts stereo 16 bit signed 16KHz audio data, outputs 8 bit unsigned mono
   audio data thru the parallel port. */
#include <OS.h>
#include <SupportDefs.h>
#include <KernelExport.h>
#include <Drivers.h>
#include <math.h>
#include <ISA.h>

#include "audio_sound_driver.h" //
#include "audio_user_IOCTL_defines.h" //

/* Sound Driver Defaults */
#define DEFAULT_SAMPLE_RATE        16000.0
#define DEFAULT_CHANNELS           2
#define DEFAULT_FORMAT             0x12 // (uchar).
#define DEFAULT_ENDIAN             0    // 0 for little endian, 1 for big endian.
#define DEFAULT_BUFFER_HEADER      0    // ??????
#define DEFAULT_WRITE_BUFFER_SIZE  B_PAGE_SIZE
#define DEFAULT_RECORD_BUFFER_SIZE B_PAGE_SIZE // 0 Should tell OS no recording available?

/* Used to load and unload driver */
status_t init_hardware (void);
status_t init_driver(void);
void uninit_driver(void);

/* Communication between BeOS & Driver */
static status_t device_open(const char *name, uint32 flags, void **cookie);
static status_t device_close(void *cookie);
static status_t device_free(void *cookie);
static status_t device_control(void *cookie, uint32 op, void *buf, size_t length);
static status_t device_read(void *cookie, off_t pos, void *buf, size_t *length);
static status_t device_write(void *cookie, off_t pos, const void *buf, size_t *length);

/* Internal functions of the driver */
static status_t set_parameters(sound_setup *setup);
static status_t get_parameters(sound_setup *setup);

static status_t play_chunk_mode_1(); // Time output with a Busy-Wait-Loop. (Historic reasons only),
static status_t play_chunk_mode_2(); // Use only Snooze_Until() to control output timing.
static status_t play_chunk_mode_3(); // Use Snooze_Until() & skip duplicate sample values.
static status_t play_chunk_mode_4(); // Snooze_Until(-Latency), Interrupt Disable, Busy Wait Loop & skip repeats.
static status_t play_chunk_mode_5( bigtime_t frame_length);    // Interrupt B_PERIODIC_TIMER.
static status_t play_chunk_mode_6( bigtime_t next_frame_time); // Interrupt B_ONE_SHOT_ABSOLUTE_TIMER fixed timing.
static status_t play_chunk_mode_7( bigtime_t next_frame_time); // Interrupt B_ONE_SHOT_ABSOLUTE_TIMER variable timing skipping dups.

static int32 device_simple_interrupt(timer *_timing, uint32 flags);
static int32 device_fixed_interrupt(timer *_timing, uint32 flags);
static int32 device_varible_interrupt(timer *_timing, uint32 flags);

/* Internal Data */
const char **publish_devices(void);
device_hooks* find_device(const char *name);

static bigtime_t next_buffer_time;  // Next time slot available to start playing a sound buffer.  Points to the
                                    // system time when next buffer recieved can be play if a buffer is
                                    // already playing.
static bool resources_flag = false; // Track if system resources are allocated.
static bool interrupt_flag = false; // Allows system interrupts during play-back of sound as default.
static bool debug = false;          // Don't send debugging information.
static bool info_write = true;      // When debug = true, dprintf() timing data.
/****************************************************************************************************************/
static int interpolate = 1; // Interpolator mode:
                            // 1 = Do not interpolate audio data.
                            // 2 = Use 2x Interpolation.
                            // 3 = Use 3x Interpolation.
                            // 4 = Use 4x Interpolation.
static int timing_mode = 2; // Set timing mode to output to parallel port.
                            // 1 = Time output with a Busy-Wait-Loop. (Historic reasons only).
                            // 2 = Use only Snooze_Until() to control output timing.
                            // 3 = Use Snooze_Until() & skip duplicate sample values.
                            // 4 = Snooze_Until(-Latency), Interrupt Disable, Busy Wait Loop & skip repeats.
                            // 5 = Interrupt B_PERIODIC_TIMER.
                            // 6 = Interrupt B_ONE_SHOT_ABSOLUTE_TIMER fixed timing.
                            // 7 = Interrupt B_ONE_SHOT_ABSOLUTE_TIMER variable timing skipping dups.
/****************************************************************************************************************/
static int timing = 0, old_timing = 0; // Records mode of the timing used to play present & previous buffers
static bigtime_t old_frame_length = 0; // sent by write_hook(data_buffer).
/****************************************************************************************************************/
static bigtime_t next_frame_time, next_buffer_time;
static long sampling_rate, frame_length, frame_rate;  // Temp globals, will change locations in future.
/****************************************************************************************************************/
static int32 open_count = 0;     // Number of openned instances of the driver. 
static int32 frame_rate = 16000; // Default sampling rate. Should be set in init_driver().
static bigtime_t Latency = 0;    // Latency to use in driver (User changable in realtime).  This defines the
                                 // maximum interrupt latency in microseconds.  We generally always want to
                                 // interrupt early and spin until the proper time.  In general, the actual
                                 // overhead for an interrupt is small, however, if other drivers have interrupts
                                 // disabled or are servicing interrupts, the actual latency could be increased.
/****************************************************************************************************************/
static sem_id write_done_sem = -1, write_lock_sem = -1; // Semaphore block writes to the driver.
static sem_id read_done_sem  = -1, read_lock_sem  = -1; // Semaphore block reads from the driver.
/****************************************************************************************************************/
static isa_module_info *isa = NULL; // Module structure to access ISA port hardware.
/****************************************************************************************************************/
#define RECORD_LENGTH DEFAULT_RECORD_BUFFER_SIZE // Size of record buffer.
static uint16 record_buffer[RECORD_LENGTH + 4];  // Buffer for 16 bit stereo data of sound to be recorded.
/****************************************************************************************************************/
#define RING_LENGTH 65536              // Size of output ring buffer.  Note: Size = 2^16.
static uchar ring_buffer[RING_LENGTH]; // Processed 8-Bit audio sound data ready to be outputted.
static uint16 read_index = 0;          // 16 Bit Pointer to read out of the ring ring_buffer.
static uint16 write_index = 0;         // 16 Bit Pointer to write into the ring ring_buffer.
/****************************************************************************************************************/
static uchar c25[65540], c33[65540], c50[65540], c66[65540], c75[65540]; // Percentage arrays for Interpolation.
/****************************************************************************************************************/
// Note:  The following varibles should normally be declared as locals or allocated from memory
// but due to bugs (lack of C++ skills) in my coding I am placing them here to help debug the code.

/* Interrupt structures needed to play the sound in the background. First the driver needs to set up a timer() for
 the first interrupt, then wait on a completion semaphore to tell it to create another timer() if the driver used
 a B_ONE_SHOT_ABSOLUTE_TIMER flag in the timer() struct.  Otherwise if the timer() is created using a flag of
 B_PERIODIC_TIMER in the timer struc it only needs to be created once but it is very important to ensure the
 timer() function is cancelled with the CancelTimer() function before the driver is closed down completely.*/
 
typedef struct { // Private Struc use to set up interrupt timing to send bytes to parllel port
 struct    timer timer;  // = { qent entry; uint16 flags; uint16 cpu; timer_hook hook; bigtime_t  period; }
 bigtime_t frame_length; // Microseconds between each sample output.
 bigtime_t next_frame;   // System_Time() of next output to be sent.
} device_timer_info;

static device_timer_info device_timer; // Create struct to hold interrupt handler info.

static audio_format AudioFormat;  // Declared to help debug code.  Should be a mallocated array.

// Storage varibles for mixer controls.
static char left_adc_source = line, right_adc_source = line, // Mixer sources to digitalize.
// Controls for channel.adc_gain 0..15 adc gain, in 1.5 dB steps.
            left_adc_gain  = 15, right_adc_gain = 15,
// Set channel.mic levels, 1bit->2bit, non-zero enables 20 dB MIC input gain.
            left_mic_gain_enable = true, right_mic_gain_enable = true,
// AUX-1 Settings - 0-31 aux1 mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes aux1 mix.
            left_aux1_mix_gain  = 31, left_aux1_mix_mute  = false,
            right_aux1_mix_gain = 31, right_aux1_mix_mute = false,
// AUX-2 Settings - 0-31 aux2 mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes aux2 mix.
            left_aux2_mix_gain  = 31, left_aux2_mix_mute  = false,
            right_aux2_mix_gain = 31, right_aux2_mix_mute = false,
// Line Levels Settings - 0-31 line mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes line mix.
            left_line_mix_gain  = 31, left_line_mix_mute  = false,
            right_line_mix_gain = 31, right_line_mix_mute = false,
// DAC Settings - 0..61 dac attenuation, in -1.5 dB steps. Non-zero mutes dac output.
            left_dac_attn  = 31, left_dac_mute  = false,
            right_dac_attn = 31, right_dac_mute = false,

            sound_sample_rate = kHz_16_0, /*  Obsolete? Only supports 16 kHz. */
            sound_playback_format = linear_16bit_little_endian_stereo, /* Obsolete? Always signed 16bit-linear. */
            sound_capture_format  = linear_16bit_little_endian_stereo, /* Obsolete? Always signed 16bit-linear. */
// Control for dither_enable, non-zero enables dither on 16 => 8 bit.
            sound_dither_enable = false,
// Control for loop_attn. 0..64 adc to dac loopback attenuation, in -1.5 dB steps. Non-zero enables loopback.
            sound_loop_attn = 31, sound_loop_enable = true,
// Control for output boost, zero (2.0 Vpp) non-zero (2.8 Vpp) output level boost.
            sound_output_boost = true,
// Control for high pass filter, non-zero enables highpass filter in adc.
            sound_highpass_enable = true,
// Control for speaker gain, 0..64 mono speaker gain. Un/Mute speaker, non-zero mutes speaker.
            sound_mono_gain = 31, sound_mono_mute = false ;

/* api_version - This varible defines the API version to which the driver was written, and should be set to
 B_CUR_DRIVER_API_VERSION at compile time. This varible's value will be changed with every revision to the driver
 API; the value with which your driver was compiled will tell devfs how it can communicate with the driver. */

int32 api_version = B_CUR_DRIVER_API_VERSION;

const char *driver_names[] = { "audio/old/sound", NULL };

/*****************************************************************************************************************
 Device Hooks - The hook functions specified in the device_hooks function returned by the driver's find_device()
 function handle requests made by devfs (and through devfs, from user applications). These are described in this
 section.  The structure itself looks like this: 
 typedef struct { 
         device_open_hook open;       -> open entry point
         device_close_hook close;     -> close entry point
         device_free_hook free;       -> free cookie
         device_control_hook control; -> control entry point
         device_read_hook read;       -> read entry point
         device_write_hook write;     -> write entry point
         device_select_hook select;   -> select entry point
         device_deselect_hook deselect; -> deselect entry point
         device_readv_hook readv;     -> posix read entry point
         device_writev_hook writev;   -> posix write entry point
 } device_hooks;
 In all cases, return B_OK if the operation is successfully completed, or an appropriate error code if not.

 Function pointers for the device hooks entry points. */

static device_hooks sound_device[] = { // NULLs=sound_select, sound_deselect, device_readv, device_writev
 device_open, device_close, device_free, device_control, device_read, device_write, NULL, NULL, NULL, NULL };

/*****************************************************************************************************************
 init_hardware - status_t init_hardware (void) - This function is called when the system is booted, which lets
 the driver detect and reset the hardware it controls. The function should return B_OK if the initialization is
 successful; otherwise, an appropriate error code should be returned. If this function returns an error, the
 driver won't be used.

 Sound Driver notes: Since this driver uses the parallel port that is already going to be initialized by the OS,
 to avoid any conflicts in the port's hardware setup we do nothing to the system hardware here.  Turns out this is
 also good location to load parameters of a driver setting file.  Note: Found out that driver_parameter string can
 not contain spaces.  Use underlines '_' if multiple words are needed.  The settings in the driver settings file
 are the same as the defaults found in this code.  They changed by just editting the 'Sound_Settings' file in the
 folder: /boot/home/config/settings/kernel/drivers/.  When the driver has finished reading in the settings and
 done anything else that is needed in init_hardware() it must return B_OK to tell the kernel that the hardware was
 detected ok, otherwise the (simulated) hardware will not be useable.  Also since an audio input is going to be
 faked, clear a buffer to read from.*/

status_t init_hardware (void) {
void *settings; // Handle to the driver settings file.
const char * str_value; // Used for string value settings.
ulong num_value; // Used for string value settings.
int index = 0;

 for (index = 0; index < 32767; index++ ) {
     record_buffer[index] = 0;  } // Fill sound buffers with zeros.

 /*if (debug)*/ dprintf("PPA: Initialize Hardware.\n");
 settings = (void *)load_driver_settings("Sound_Settings"); // Open the settings file for this driver.
            // It is located in ~/config/settings/kernel/drivers.

 if (settings) { // Test file exists.  If file does not exist varible will equal zero(0).
    /*if (debug)*/ dprintf("PPA: Driver's Settings File Found.\n");

    if (debug) { dprintf("PPA: Debug = true.\n"); } else { dprintf("PPA: Debug = false.\n"); }
    debug = get_driver_boolean_parameter(settings, "Debug_Info:", false, false); // Debugging status.
    if (debug) { dprintf("PPA: Debug = true.\n"); } else { dprintf("PPA: Debug = false.\n"); }
            // If debug = True Enables dprint() of debugging info for details of driver operation.
            // If the parameter was not found, the third parameter 'false' is returned.
            // If the parameter was found, but no value defined, the fourth parameter 'false' is returned.
/*--------------------------------------------------------------------------------------------------------------*/
    interrupt_flag = get_driver_boolean_parameter(settings, "System_Interrupts:", false, false); // Interrupt status.
    if (debug) {
       if (interrupt_flag) { 
          dprintf("PPA: Interrupts not allowed during playing of sounds.\n");
       } else {
          dprintf("PPA: Interrupts are allowed during playing of sounds.\n"); } }
            // False = System Interrupts allowed.  True = System Interrupts disabled.
            // If the parameter was not found, the third parameter 'false' is returned.
            // If the parameter was found, but no value defined, the fourth parameter 'false' is returned.
/*--------------------------------------------------------------------------------------------------------------*/
    str_value = (const char*)get_driver_parameter(settings, "Interpolation:", "Unknown", "NoArg");
            // If the parameter was not found, the third parameter 'Unknown' is returned.
            // If the parameter was found, but no value defined, the fourth parameter 'NoArg' is returned.
    if (strcmp(str_value,"Unknown") != 0) { // Test if parameter was found.
         if (strcmp(str_value, "NoArg") != 0) { // Test if unknown value for parameter.
            num_value = atoi(str_value); // Convert to a integer number.
            if ((num_value > 0) && (num_value < 5)) { // Test in correct range of values.
               interpolate = num_value;
               if (debug) dprintf("PPA: Interpolation set to %d .\n", (long)interpolate); // Valid value.
            } else {
               if (debug) dprintf("PPA: Interpolation parameter value out of range.\n"); }
         } else {
            if (debug) dprintf("PPA: Interpolation parameter value missing.\n"); }
    } else {
       if (debug) dprintf("PPA: Interpolation parameter not found.\n"); }
/*--------------------------------------------------------------------------------------------------------------*/
    str_value = (const char*)get_driver_parameter(settings, "Sampling_Rate:", "Unknown", "NoArg");
            // If the parameter was not found, the third parameter 'Unknown' is returned.
            // If the parameter was found, but no value defined, the fourth parameter 'NoArg' is returned.
    if (strcmp(str_value,"Unknown") != 0) { // Test if parameter was found.
         if (strcmp(str_value, "NoArg") != 0) { // Test if unknown value for parameter.
            num_value = atoi(str_value); // Convert to a integer number.
            if ((num_value > 3999) && (num_value < 100001)) { // Test in correct range of values.
               frame_rate = num_value;
               if (debug) dprintf("PPA: Sampling Rate %d .\n", (long)frame_rate); // Valid value.
            } else {
               if (debug) dprintf("PPA: Sampling Rate parameter value out of range.\n"); }
         } else {
            if (debug) dprintf("PPA: Sampling Rate parameter value missing.\n"); }
    } else {
       if (debug) dprintf("PPA: Sampling Rate parameter not found.\n"); }
/*--------------------------------------------------------------------------------------------------------------*/
    str_value = (const char*)get_driver_parameter(settings, "Timing_Mode:", "Unknown", "NoArg");
            // If the parameter was not found, the third parameter 'Unknown' is returned.
            // If the parameter was found, but no value defined, the fourth parameter 'NoArg' is returned.
    if (strcmp(str_value,"Unknown") != 0) { // Test if parameter was found.
         if (strcmp(str_value, "NoArg") != 0) { // Test if unknown value for parameter.
            num_value = atoi(str_value); // Convert to a integer number.
            if ((num_value > 0) && (num_value < 8)) { // Test in correct range of values.
               timing_mode = num_value;
               if (debug) dprintf("PPA: Timing Mode %d .\n", (long)timing_mode); // Valid value.
            } else {
               if (debug) dprintf("PPA: Timing Mode parameter value out of range.\n"); }
         } else {
            if (debug) dprintf("PPA: Timing Mode parameter value missing.\n"); }
    } else {
       if (debug) dprintf("PPA: Timing Mode parameter not found.\n"); }
/*--------------------------------------------------------------------------------------------------------------*/
    str_value = (const char*)get_driver_parameter(settings, "Timing_Latency:", "Unknown", "NoArg");
            // If the parameter was not found, the third parameter 'Unknown' is returned.
            // If the parameter was found, but no value defined, the fourth parameter 'NoArg' is returned.
    if (strcmp(str_value,"Unknown") != 0) { // Test if parameter was found.
         if (strcmp(str_value, "NoArg") != 0) { // Test if unknown value for parameter.
            num_value = atoi(str_value); // Convert to a integer number.
            if ((num_value >= 0) && (num_value < 21)) { // Test in correct range of values.
               Latency = num_value;
               if (debug) dprintf("PPA: Latency Timing %d .\n", (long)Latency); // Valid value.
            } else {
               if (debug) dprintf("PPA: Latency Timing parameter value out of range.\n"); }
         } else {
            if (debug) dprintf("PPA: Latency Timing parameter value missing.\n"); }
    } else {
       if (debug) dprintf("PPA: Latency Timing parameter not found.\n"); }

 } else {
    dprintf("PPA: Setting File *NOT* Found..\n"); }

 unload_driver_settings(settings); // Always close the settings file before exiting.
 return B_OK; } 

/*****************************************************************************************************************
 init_driver - status_t init_driver(void) - optional function - called every time the driver is loaded.  Drivers
 are loaded and unloaded on an as-needed basis. When a driver is loaded by devfs, this function is called to let
 the driver allocate memory and other needed system resources. Return B_OK if initialization succeeds, otherwise
 return an appropriate error code. <<<what happens if this returns an error?>>> <<<what happens if this returns
 an error? Appears to unload driver.  Note, if the code detects an error allocating a resource, it frees up all
 previous allocated resources before returning the error of the resource that failed. >>>
 
 Sound Driver notes: Write_Lock's purpose is to lock out any additional writes to the driver before it has
 finished servicing the previous write to the driver.  Read_Lock serves the same purpose for any reads from the
 driver.  In earlier drivers Write_Lock was not released untill all the sound data was played.  In this driver a
 large ring buffer holds the data instead and Write_Lock will be released as soon as all the audio data has been
 transfered to said ring buffer.  This releases the driver a lot earlier to recieve more data.

 Again since the parallel port that is already going to be initialized by the OS, this driver does not try to do
 any parallel port hardware setup here. */

status_t init_driver(void) {
 int32 old_data = 0, new_data = 0; status_t ret = B_OK;
 if (debug) dprintf("PPA: Initializing Driver.\n");

 while ( old_data < 256 ) {
        while ( new_data < 256) { // Create Convertion data arrays for over-sampling format.
              c25[old_data*256+new_data] = (uchar)((3 * old_data + new_data    ) / 4); // (Old*3+New*1)/4 = 25%
              c33[old_data*256+new_data] = (uchar)((2 * old_data + new_data    ) / 3); // (Old*2+New*1)/3 = 33%
              c50[old_data*256+new_data] = (uchar)((    old_data + new_data    ) / 2); // (Old*1+New*1)/2 = 50%
              c66[old_data*256+new_data] = (uchar)((    old_data + new_data * 2) / 3); // (Old*1+New*2)/3 = 66%
              c75[old_data*256+new_data] = (uchar)((    old_data + new_data * 3) / 4); // (Old*1+New*3)/4 = 75%
              new_data++; }; new_data = 0; old_data++; }

 ret = get_module(B_ISA_MODULE_NAME, (module_info**) &isa);
 if (ret == B_OK) {
    if (debug) dprintf("PPA: B_ISA_MODULE_NAME ok.\n");

    write_lock_sem = create_sem(1, "write lock sem"); ret = write_lock_sem;
    if (write_lock_sem > B_OK) {
       if (debug) dprintf("PPA: Created write_lock sem ok.\n");

       write_done_sem = create_sem(0, "write finished"); ret = write_done_sem;
       if (write_done_sem > B_OK) {
          if (debug) dprintf("PPA: Created write_done_sem ok.\n");

           read_lock_sem  = create_sem(1, "read  lock"); ret = read_lock_sem;
           if (read_lock_sem > B_OK) {
              if (debug) dprintf("PPA: Created read_lock_sem ok.\n");

              read_done_sem = create_sem(0, "read finished"); ret = write_done_sem;
              if (read_done_sem > B_OK) {
                 if (debug) dprintf("PPA: Created read_done_sem ok.\n");
                 resources_flag = true; return B_OK;  // No errors creating all resources.

              } else {
                 if (debug) dprintf("PPA: Failed to create read_done_sem."); }
              delete_sem(read_lock_sem); if (debug) dprintf("PPA: Deleted read_lock_sem.\n");

           } else {
              if (debug) dprintf("PPA: Failed to create read_done_sem."); }
           delete_sem(write_done_sem); if (debug) dprintf("PPA: Deleted write_done_sem.\n");

        } else {
          if (debug) dprintf("PPA: Failed to create write_done_sem.\n"); }
        delete_sem(write_lock_sem); if (debug) dprintf("PPA: Deleted write_lock_sem.\n");

     } else {
        if (debug) dprintf("PPA: Failed to create write_lock_sem."); }
     put_module(B_ISA_MODULE_NAME); if (debug) dprintf("PPA: Deleted B_ISA_MODULE_NAME.\n"); 
 
 } else {
     if (debug) dprintf("PPA: Failed to get B_ISA_MODULE_NAME.\n"); }

 if (debug) dprintf("\n"); // Spacer in debugger output.
 return ret; }

/*****************************************************************************************************************
 uninit_driver - void uninit_driver(void) - optional function - called every time the driver is unloaded.  This
 function is called by devfs just before the driver is unloaded from memory. This lets the driver clean up after
 itself, freeing any resources it allocated.

 Sound Driver notes: Since this driver uses the parallel port that is already going to be initialized by the OS,
 to avoid any conflicts in the port's hardware setup we do nothing here.  Also uninit_driver() does not return a
 status report on the freeing up of resources, you should not free up something here unless you are 100% sure that
 it will happen. */

void uninit_driver(void) {
 if (debug) dprintf("PPA: Uninitializing Driver.\n\n");
 return; }

/*****************************************************************************************************************
 open_hook() - status_t open_hook(const char *name, uint32 flags, void **cookie) - This hook function is called
 when a program opens one of the devices supported by the driver. The name of the device (as returned by
 publish_devices()) is passed in name, along with the flags passed to the Posix open() function. Cookie points to
 space large enough for you to store a single pointer. You can use this to store state information specific to the
 open() instance. If you need to track information on a per-open() basis, allocate the memory you need and store a
 pointer to it in **cookie.

 Sound Driver notes: I have had real problems using the cookie structure.  Normally you would allocate a block of
 memory to hold the Struc 'AudioFormat' and use the cookie to point to that Struc.  However whenever I tried to do
 this I kept running into problems.  I don't know what I was doing wrong but to get the code running I created
 'AudioFormat' in a Global Struc and used that instead.  Since the cookie pointer is not really needed in this
 driver I just set the cookie to NULL and never reference it again.  That is why you don't see me use it anywhere.
 Notice also that I do not limit the value of Open_Count, many drivers do limit the system to openning a driver
 to one open instance only.  This helps avoid conflicts, in this case however, I wanted the ability to send IOCTL
 codes to the driver indepentantly of the sound playing.  Since I added the ability to control different timing
 options using a re-write of Alexander G. M. Smith's AGMSDeviceTest program (called Sound_Controller) it became
 necessary to have the ability for the driver to be openned by more than one accessing thread at the same time. */

static status_t device_open(const char *name, uint32 flags, void **cookie) {
 *cookie = NULL;  // Cookie not used.

 if (debug) dprintf("PPA: Open Driver.\n\n"); 

 if (atomic_add(&open_count, 1) < 1) { // Increase count of open instances. Confirmed first time openned.
    AudioFormat.sample_rate        = DEFAULT_SAMPLE_RATE;
    AudioFormat.channels           = DEFAULT_CHANNELS;
    AudioFormat.format             = DEFAULT_FORMAT;
    AudioFormat.big_endian         = DEFAULT_ENDIAN;
    AudioFormat.buffer_header      = DEFAULT_BUFFER_HEADER;
    AudioFormat.write_buffer_size  = DEFAULT_WRITE_BUFFER_SIZE;
    AudioFormat.record_buffer_size = DEFAULT_RECORD_BUFFER_SIZE;
    if (debug) {
       dprintf("PPA: OLD AudioFormat Address = %lx \n",&AudioFormat);
       dprintf("     New sample_rate = %d \n",(long)AudioFormat.sample_rate);
       dprintf("         sample_rate = %f \n",(double)AudioFormat.sample_rate);
       dprintf("         sample_rate = %f \n",(float)AudioFormat.sample_rate);
       dprintf("     New channels = %d \n",AudioFormat.channels);
       dprintf("     New format = %d \n",AudioFormat.format);
       dprintf("     New big_endian = %d \n",AudioFormat.big_endian);
       dprintf("     New buffer_header = %d \n",AudioFormat.buffer_header);
       dprintf("     New write_buffer_size = %d \n",AudioFormat.write_buffer_size);
       dprintf("     New record_buffer_size = %d \n\n",AudioFormat.record_buffer_size); }
 } else { // Driver was already openned.
    if (debug) dprintf("PPA: Driver opened already.\n\n"); }

 return B_OK; }

/*****************************************************************************************************************
 close_hook() - status_t close_hook(void *cookie) - This hook is called when an open instance of the driver is
 closed using the close() Posix function. Note that because of the multithreaded nature of the BeOS, it's possible
 there may still be transactions pending, and you may receive more calls on the device. For that reason, you
 shouldn't free instance-wide system resources here. Instead, you should do this in free_hook(). However, if there
 are any blocked transactions pending, you should unblock them here.

 Sound Driver notes: Originally Open_Count was not used in this driver, but because I needed to open it also with
 the program 'Sound_Controller' the code was changed.  Since it is possible to have more than one open instance,
 I do not free resources here unless this is the final close_hook() call.  Free_Hook is the place to normally do
 this, I just added it to the code here to show how to do it and to be safe. */

static status_t device_close(void *cookie) {
 if (debug) dprintf("PPA: Closed the Driver.\n");
 atomic_add(&open_count, -1); // Decrease count on number of open instances.
 if (open_count == 0) { // Added to make sure resources are freed at the right time only.
    if (resources_flag == true) { // Only free resources if they still exist.
       delete_sem(write_done_sem); delete_sem(write_lock_sem); // Delete write locks.
       if (debug) dprintf("PPA: Deleted write semaphores.\n");
       delete_sem(read_done_sem);  delete_sem(read_lock_sem);  // Delete read locks.
       if (debug) dprintf("PPA: Deleted read semaphores.\n");
       put_module(B_ISA_MODULE_NAME); resources_flag = false;  // Free resources and tag status.
       if (debug) dprintf("PPA: Deleted B_ISA_MODULE_NAME.\n");
       if (timing > 4) {                                   // Test if an interrupt timer is active.
          timing = 0;
          while (write_index != read_index) { snooze(1000); }  // Delay until ring_buffer is empty.
          cancel_timer((timer*)&device_timer); } } } // Cancel timer().
 if (debug) dprintf("\n"); // Spacer in debugger output.
 return B_OK; }

/*****************************************************************************************************************
 free_hook() - status_t free_hook(void *cookie) - This hook is called once all pending transactions on an open
 (but closing) instance of your driver are completed. This is where your driver should release instance wide
 system resources. free_hook() doesn't correspond to any Posix function.

 Sound Driver notes: Most sample code I found free up the driver's resources here.  When I tested the calls to the
 driver I found calls to free_hook() worked all the time, calls to uninit_driver() did not happen when I expected
 them to.  It may be best to have the same code in both driver functions.
 
 In the case of the parallel port's hardware setup we do nothing here. */

static status_t device_free(void *cookie) {
 if (debug) dprintf("PPA: Free the Driver.\n");
 if (open_count < 2) { // Do not free resources if driver still open to more than one caller.
    if (resources_flag == true) { // Only free resources if they still exist.
       delete_sem(write_done_sem); delete_sem(write_lock_sem); // Delete write locks.
       if (debug) dprintf("PPA: Deleted write semaphores.\n");
       delete_sem(read_done_sem);  delete_sem(read_lock_sem);  // Delete read locks.
       if (debug) dprintf("PPA: Deleted read semaphores.\n");
       put_module(B_ISA_MODULE_NAME); resources_flag = false;  // Free resources and tag status.
       if (debug) dprintf("PPA: Deleted B_ISA_MODULE_NAME.\n");
       if (timing > 4) {                                   // Test if an interrupt timer is active.
          timing = 0;
          while (write_index != read_index) { snooze(1000); }  // Delay until ring_buffer is empty.
          cancel_timer((timer*)&device_timer); } } } // Cancel timer().
 if (debug) dprintf("\n"); // Spacer in debugger output.
 return B_OK; }

/*****************************************************************************************************************
 control_hook() - status_t control_hook(void *cookie, uint32 op, void *data, size_t length) - This hook handles
 the ioctl() function for an open instance of your driver. The control hook provides a means to perform operations
 that don't map directly to either read() or write(). It receives the cookie for the open instance, plus the
 command code op and the data and len arguments specified by ioctl()'s caller.  These arguments have no inherent
 relationship; they're simply arguments to ioctl() that are forwarded to your hook function. Their definitions are
 defined by the driver. Common command codes can be found in be/drivers/Drivers.h.  The len argument is only valid
 when ioctl() is called from user space; the kernel always sets it to 0.

 Sound Driver notes: Using another re-write of Alexander G. M. Smith's AGMSDeviceTest program
 (now called 'Sound_Controller') diffirent control codes can be sent to the driver to change the diffirent ways
 the sound driver works so that diffirent options can be compared. */

static status_t device_control(void *cookie, uint32 opcode, void *data_buffer, size_t length) {
 status_t err; audio_buffer_header *header; size_t data_length;
 static float rates[1] = {DEFAULT_SAMPLE_RATE }; // Works for me as far as testing goes.
// static float rates[5] = { 32000.0, 22050.0, 16000.0, 11025.0, 8000.0 };

 if (debug) dprintf("PPA: OpCode.\n");
 switch(opcode) {

    case USER_PLAIN_AUDIO: // Plain Audio.  Outputs audio sample data directly to the DAC.
         if (debug) dprintf("PPA: Disabled Interpolation.  Outputs plain audio data.\n\n");
         interpolate = 1; return B_OK; // Allow new calls to device_write.

    case USER_INTERPOLATOR_x2: // Interpolated Audio.  Outputs audio sample data x2 times.
         if (debug) dprintf("PPA: Enabled Interpolation.  Outputs 2x interpolated audio data.\n\n");
         interpolate = 2; return B_OK;

    case USER_INTERPOLATOR_x3: // Interpolated Audio.  Outputs audio sample data x3 times faster.
         if (debug) dprintf("PPA: Enabled Interpolation.  Outputs 3x interpolated audio data.\n\n");
         interpolate = 3; return B_OK;

    case USER_INTERPOLATOR_x4: // Interpolated Audio.  Outputs audio sample data x4 times.
         if (debug) dprintf("PPA: Enabled Interpolation.  Outputs 4x interpolated audio data.\n\n");
         interpolate = 4; return B_OK;
/*--------------------------------------------------------------------------------------------------------------*/
    case USER_INTERRUPTS_YES: // Allows system interrupts during play-back of sound.
         if (debug) dprintf("PPA: Enabled system interrupts.\n\n");
         interrupt_flag = false; return B_OK;
    case USER_INTERRUPTS_NO: // Disable system interrupts during play-back of sound.
         if (debug) dprintf("PPA: Disabled system interrupts.\n\n");
         interrupt_flag = true; return B_OK;
/*--------------------------------------------------------------------------------------------------------------*/
    case USER_BUSY_WAIT_TIMING: // Use busy wait loops to control timing.
         if (debug) dprintf("PPA: Use Busy Wait Loop.\n\n");
         timing_mode = 1; return B_OK;

    case USER_SNOOZE_UNTIL_TIMING: // Use snooze_until() function to control timing.
         if (debug) dprintf("PPA: Use snooze_until().\n\n");
         timing_mode = 2; return B_OK;

    case USER_SKIP_DUP_TIMING: // Use snooze_until() function to control timing and skip repeats.
         if (debug) dprintf("PPA: Use snooze_until() to skip duplicates in sample outputs.\n\n");
         timing_mode = 3; return B_OK;

    case USER_LATENCY_TIMING: // Use Snooze_Until(-Latency), Interrupt Disable, Busy Wait Loop & skip repeats.
         if (debug) dprintf("PPA: Use snooze_until() to skip duplicates in sample outputs.\n\n");
         timing_mode = 4; return B_OK;

    case USER_PERIODIC_TIMER: // Use Interrupt Timer(B_PERIODIC_TIMER) to control timing.
         if (debug) dprintf("PPA: Use B_PERIODIC_TIMER.\n\n");
         timing_mode = 5; return B_OK;

    case USER_ONE_SHOT_ABSOLUTE_TIMER: // Use Timer(B_ONE_SHOT_ABSOLUTE_TIMER) to control timing.
         if (debug) dprintf("PPA: Use B_ONE_SHOT_ABSOLUTE_TIMER.\n\n");
         timing_mode = 6; return B_OK;

    case USER_ONE_SHOT_SKIP_TIMER: // Use Timer(B_ONE_SHOT_ABSOLUTE_TIMER) to control timing.
         if (debug) dprintf("PPA: Use B_ONE_SHOT_ABSOLUTE_TIMER to skip duplicates.\n\n");
         timing_mode = 7; return B_OK;
/*--------------------------------------------------------------------------------------------------------------*/
    case USER_LATENCY_00: // Use timing with a Latency = 0 Microseconds.
         if (debug) dprintf("PPA: Latency = 0.\n\n");
         Latency = (bigtime_t)0; return B_OK;
    case USER_LATENCY_05: // Use timing with a Latency of 5 Microseconds.
         if (debug) dprintf("PPA: Latency = 5.\n\n");
         Latency = (bigtime_t)5; return B_OK;
    case USER_LATENCY_10: // Use timing with a Latency of 10 Microseconds.
         if (debug) dprintf("PPA: Latency = 10.\n\n");
         Latency = (bigtime_t)10; return B_OK;
    case USER_LATENCY_20: // Use timing with a Latency of 10 Microseconds.
         if (debug) dprintf("PPA: Latency = 20.\n\n");
         Latency = (bigtime_t)20; return B_OK;
/*--------------------------------------------------------------------------------------------------------------*/
    case USER_DEBUG_OFF: // Disables dprint() of debugging info.
         dprintf("PPA: Debug Info Turned Off.\n\n");
         debug = false; return B_OK;
    case USER_DEBUG_ON: // Enables dprint() of debugging info for details of driver operation.
         dprintf("PPA: Debug Info Turned On.\n\n");
         debug = true; return B_OK;
/*--------------------------------------------------------------------------------------------------------------*/
    case B_GET_DEVICE_SIZE: // Returns a size_t (32 bits) indicating the device size in bytes.
         if (debug) dprintf("PPA: Get device size.\n\n");
         *(size_t*)data_buffer = AudioFormat.write_buffer_size; return B_OK;
    case B_SET_DEVICE_SIZE: // Sets the device size to the value pointed to by data.
         if (debug) dprintf("PPA: Set device size.\n\n");
         // AudioFormat.write_buffer_size = *(size_t*)data_buffer;
         return B_OK;
/*--------------------------------------------------------------------------------------------------------------*/
    case B_SET_NONBLOCKING_IO: // Sets the device to use nonblocking I/O.  Always run in blocking mode.
         if (debug) dprintf("PPA: Set to non-blocking I/O.\n\n");
         return B_ERROR;
    case B_SET_BLOCKING_IO: // Sets the device to use blocking I/O.
         if (debug) dprintf("PPA: Set to blocking I/O.\n\n");
         return B_OK;
    case B_GET_READ_STATUS: // Returns 'true' if the device can read without blocking, otherwise 'false'.
         if (debug) dprintf("PPA: Get Read Status.\n\n");
         *(bool*)data_buffer = false; return B_OK; // Always run in blocking mode.
    case B_GET_WRITE_STATUS: // Returns 'true' if the device can write without blocking, otherwise 'false'.
         if (debug) dprintf("PPA: Get Write Status.\n\n");
         *(bool*)data_buffer = false; return B_OK; // Always run in blocking mode.
/*--------------------------------------------------------------------------------------------------------------*/
    case B_GET_GEOMETRY: // Fills out the specified device_geometry structure to describe the device.
         if (debug) dprintf("PPA: Get Drive Geometry.\n\n"); // Not a drive, return a error.
         return B_ERROR;

    case B_GET_DRIVER_FOR_DEVICE: // Returns the path of the driver executable handling the device.
         if (debug) dprintf("PPA: Get Device Path.\n\n");
         //*(char*)data_buffer = "/boot/home/config/add-ons/kernel/drivers/bin/sound"; 
         return B_OK;

    case B_GET_PARTITION_INFO: // Returns a partition_info structure for the device.
         if (debug) dprintf("PPA: Get Partition Info.\n\n");
         return B_ERROR;
    case B_SET_PARTITION: // Creates a user-defined partition. data points to a partition_info structure.
         if (debug) dprintf("PPA: Set Partition.\n\n");
         return B_ERROR;

    case B_FORMAT_DEVICE: // Formats the device. data should point to a boolean value. If this is  true , the  device is formatted low-level. If it's  false , <<<unclear>>>
         if (debug) dprintf("PPA: Format Device.\n\n");
         return B_OK;

    case B_EJECT_DEVICE: // Ejects the device.
         if (debug) dprintf("PPA: Eject Device.\n\n");
         return B_OK;

    case B_GET_ICON: // Fills out the specified device_icon structure to describe the device's icon.
         if (debug) dprintf("PPA: Get Icon.\n\n");
         return B_ERROR;

    case B_GET_BIOS_GEOMETRY: // Fills out a device_geometry structure to describe the device as the BIOS sees it.
         if (debug) dprintf("PPA: .\n\n");
         return B_ERROR;

    case B_GET_MEDIA_STATUS: // Gets the status of the media in the device by placing a status_t at the location pointed to by data.
         if (debug) dprintf("PPA: .\n\n");
         return B_OK;

    case B_LOAD_MEDIA: // Loads the media, if this is supported. <<<what does that mean?>>>
         if (debug) dprintf("PPA: Load Media.\n\n");
         return B_OK;

    case B_GET_BIOS_DRIVE_ID: // Returns the BIOS ID for the device.
         if (debug) dprintf("PPA: Get BIOS ID.\n\n");
         return B_ERROR;
/*--------------------------------------------------------------------------------------------------------------*/
    case B_SET_UNINTERRUPTABLE_IO: // Prevents control-C from interrupting I/O.
         if (debug) dprintf("PPA: Set UnInterruptable I/O.\n\n");
         return B_OK;
    case B_SET_INTERRUPTABLE_IO: // Allows control-C to interrupt I/O.
         if (debug) dprintf("PPA: Set UnInterruptable I/O.\n\n");
         return B_OK;
/*--------------------------------------------------------------------------------------------------------------*/
    case B_FLUSH_DRIVE_CACHE: // Flushes the drive's cache.
         if (debug) dprintf("PPA: Flush Cache.\n\n");
         read_index = 0;  // Pointer to read out of the ring ring_buffer.
         write_index = 0; // Pointer to write into the ring ring_buffer.
         return B_OK;
/*--------------------------------------------------------------------------------------------------------------*/
    case B_GET_NEXT_OPEN_DEVICE: // Iterates through open devices; data points to an open_device_iterator.
         if (debug) dprintf("PPA: Get Next Open Device.\n\n");
         return B_ERROR;

    case B_ADD_FIXED_DRIVER: // For internal use only.
         if (debug) dprintf("PPA: Add Fixed Driver.\n\n");
         return B_OK;
    case B_REMOVE_FIXED_DRIVER: // For internal use only.
         if (debug) dprintf("PPA: Remove Fixed Driver.\n\n");
         return B_OK;
/*--------------------------------------------------------------------------------------------------------------*/
    case SOUND_GET_PARAMS: // Copies parameters into a sound struct pointed to inside the data buffer.
         // dprintf("PPA: GET PARAMETERS\n");
         return get_parameters((sound_setup *)data_buffer);

    case SOUND_SET_PARAMS: // Copies parameters from a sound struct pointed to inside the data buffer.
         // dprintf("PPA: SET PARAMETERS\n");
         return set_parameters((sound_setup *)data_buffer);

    case B_AUDIO_SET_AUDIO_FORMAT: // arg = ptr to struct audio_format
         if (debug) {
            dprintf("PPA: SET AUDIO FORMAT\n");
            dprintf("Old sample_rate = %d \n",(long)AudioFormat.sample_rate);
            dprintf("Old channels = %d \n",AudioFormat.channels);
            dprintf("Old format = %d \n",AudioFormat.format);
            dprintf("Old big_endian = %d \n",AudioFormat.big_endian);
            dprintf("Old buffer_header = %d \n",AudioFormat.buffer_header);
            dprintf("Old write_buffer_size = %d \n",AudioFormat.write_buffer_size);
            dprintf("Old record_buffer_size = %d \n",AudioFormat.record_buffer_size); };
         memcpy(&AudioFormat, data_buffer, min(sizeof(audio_format), length));
         if (debug) {
            dprintf("Set sample_rate = %d \n",(long)AudioFormat.sample_rate);
            dprintf("Set channels = %d \n",AudioFormat.channels);
            dprintf("Set format = %d \n",AudioFormat.format);
            dprintf("Set big_endian = %d \n",AudioFormat.big_endian);
            dprintf("Set buffer_header = %d \n",AudioFormat.buffer_header);
            dprintf("Set write_buffer_size = %d \n",AudioFormat.write_buffer_size);
            dprintf("Set record_buffer_size = %d \n\n",AudioFormat.record_buffer_size); };
         return B_OK;

    case B_AUDIO_GET_AUDIO_FORMAT: // arg = ptr to struct audio_format
         if (debug) {
            dprintf("PPA: GET AUDIO FORMAT\n");
            dprintf("Get sample_rate = %d \n",(long)AudioFormat.sample_rate);
            dprintf("Get channels = %d \n",AudioFormat.channels);
            dprintf("Get format = %d \n",AudioFormat.format);
            dprintf("Get big_endian = %d \n",AudioFormat.big_endian);
            dprintf("Get buffer_header = %d \n",AudioFormat.buffer_header);
            dprintf("Get write_buffer_size = %d \n",AudioFormat.write_buffer_size);
            dprintf("Get record_buffer_size = %d \n\n",AudioFormat.record_buffer_size); };
         memcpy(data_buffer, &AudioFormat, sizeof(audio_format)); return B_OK;

    case B_AUDIO_GET_PREFERRED_SAMPLE_RATES: // arg = ptr to float[4]
         if (debug) {
            dprintf("PPA: GET PREFERRED SAMPLE RATES\n");
            dprintf("Rate 0 = %d \n",(long)rates[0]); dprintf("Rate 1 = %d \n",(long)rates[1]);
            dprintf("Rate 2 = %d \n",(long)rates[2]); dprintf("Rate 3 = %d \n\n",(long)rates[3]); };
         memcpy(data_buffer, rates, sizeof(rates)); return B_OK; 

    case SOUND_WRITE_BUFFER: // Play sound sample attached to audio_buffer_header.
         if (debug) dprintf("PPA: WRITE BUFFER\n");
         header = (audio_buffer_header *)data_buffer;        // Assign audio_buffer_header struct to data buffer.
         data_length = header->reserved_1 - sizeof(*header); // Sound data size after audio_buffer_header struct.
         if ( next_buffer_time < system_time() ) {           // First sample out or a long silence occurred.
            next_buffer_time = system_time(); }
//            long sampling_rate, frame_length;
//            if (debug) dprintf("PPA: next_buffer_time is wrong.\n");
//            sampling_rate = (long)(AudioFormat.sample_rate); // Samples per second to be sent to parallel port.
//            frame_length = 1000000L / sampling_rate;         // How many (integer) microseconds between samples.
//            next_buffer_time = (bigtime_t)((1+(system_time()/frame_length))*frame_length); } // Quantum Timing.
         header->time = next_buffer_time;                    // Time this buffer's data will start playing.
         header->sample_clock = next_buffer_time;            // Number of microseconds to play one second of sound?
         device_write(cookie, 0, header + 1, &data_length);  // Play sound data.
         return B_OK;                                        // Release semaphore to help debugging.
 
    case SOUND_READ_BUFFER:
         if (debug) dprintf("PPA: READ BUFFER\n");
         header = (audio_buffer_header *)data_buffer;       // Assign audio_buffer_header struct to data buffer.
         data_length = header->reserved_1 - sizeof(*header); // Sound data size after audio_buffer_header struct.
         header->time = system_time();                      // Time this buffer's data will start recording.
         header->sample_clock = system_time();              // Number of microseconds to play one second of sound?
         device_read(cookie, 0, header + 1, &data_length);  // Record sound data.
         release_sem(read_done_sem); return B_OK;           // Release semaphore to help debugging.

    case SOUND_SET_PLAYBACK_COMPLETION_SEM:
         if (debug) dprintf("PPA: SET PLAYBACK SEM\n");
         write_done_sem = *((sem_id *)data_buffer); return B_OK;

    case SOUND_SET_CAPTURE_COMPLETION_SEM:
         if (debug) dprintf("PPA: SET CAPTURE SEM\n\n");
         read_done_sem = *((sem_id *)data_buffer); return B_OK;

    case SOUND_SET_CAPTURE_PREFERRED_BUF_SIZE:
         if (debug) dprintf("PPA: SET CAPTURE SIZE\n");
         if (debug) dprintf("Old record_buffer_size = %d \n",AudioFormat.record_buffer_size);
         AudioFormat.record_buffer_size = (int32)data_buffer;
         if (debug) dprintf("Set record_buffer_size = %d \n\n",AudioFormat.record_buffer_size);
         return B_OK;

    case SOUND_SET_PLAYBACK_PREFERRED_BUF_SIZE:
         if (debug) dprintf("PPA: SET PLAYBACK SIZE\n");
         if (debug) dprintf("Old write_buffer_size = %d \n",AudioFormat.write_buffer_size);
         AudioFormat.write_buffer_size = (int32)data_buffer;
         if (debug) dprintf("Set write_buffer_size = %d \n\n",AudioFormat.write_buffer_size);
         return B_OK;

    case SOUND_GET_CAPTURE_PREFERRED_BUF_SIZE:
         if (debug) dprintf("PPA: GET CAPTURE SIZE\n");
         if (debug) dprintf("Get record_buffer_size = %d \n\n",AudioFormat.record_buffer_size);
         *(int32*)data_buffer = AudioFormat.record_buffer_size; return B_OK;

    case SOUND_GET_PLAYBACK_PREFERRED_BUF_SIZE:
         if (debug) dprintf("PPA: GET GET_PLAYBACK SIZE\n");
         if (debug) dprintf("Get write_buffer_size = %d \n\n",AudioFormat.write_buffer_size);
         *(int32*)data_buffer = AudioFormat.write_buffer_size; return B_OK;
 
    case B_ROUTING_GET_VALUES: // arg = ptr to struct audio_routing_cmd
         if (debug) dprintf("PPA: B_ROUTING_GET_VALUES Command.\n\n"); return B_ERROR;

    case B_ROUTING_SET_VALUES: // arg = ptr to struct audio_routing_cmd
         if (debug) dprintf("PPA: B_ROUTING_SET_VALUES Command.\n\n"); return B_ERROR;

    case B_MIXER_GET_VALUES: // arg = ptr to struct audio_routing_cmd
         if (debug) dprintf("PPA: B_MIXER_GET_VALUES Command.\n\n"); return B_ERROR;

    case B_MIXER_SET_VALUES: // arg = ptr to struct audio_routing_cmd
         if (debug) dprintf("PPA: B_MIXER_SET_VALUES Command.\n\n"); return B_ERROR;

    case B_AUDIO_GET_TIMING: // arg = ptr to struct audio_timing, used to be SV_SECRET_HANDSHAKE (10100)
         if (debug) dprintf("PPA: B_AUDIO_GET_TIMING Command.\n\n");
         return B_ERROR;

    case SOUND_GET_PLAYBACK_TIMESTAMP:
         if (debug) dprintf("PPA: SOUND_GET_PLAYBACK_TIMESTAMP Command.\n\n");
         return B_ERROR;

    case SOUND_GET_CAPTURE_TIMESTAMP:
         if (debug) dprintf("PPA: SOUND_GET_CAPTURE_TIMESTAMP Command.\n\n");
         return B_ERROR;

    case SOUND_DEBUG_ON:
         if (debug) dprintf("PPA: SOUND_DEBUG_ON Command.\n\n");
         return B_ERROR;

    case SOUND_DEBUG_OFF:
         if (debug) dprintf("PPA: SOUND_DEBUG_OFF Command.\n\n");
         return B_ERROR;

    case SOUND_LOCK_FOR_DMA:
         if (debug) dprintf("PPA: SOUND_LOCK_FOR_DMA Command.\n\n");
         return B_ERROR;

    default:
         if (debug) dprintf("PPA: Unknown Command Recieved=> %d \n\n",(int)opcode);
} return EINVAL; }
/*--------------------------------------------------------------------------------------------------------------*/
status_t set_parameters(sound_setup *sound) {
   if (debug) dprintf("PPA: set_parameters.\n");
// Mixer sources to digitalize.
   left_adc_source = sound->left.adc_source; right_adc_source = sound->right.adc_source;
// Controls for channel.adc_gain 0..15 adc gain, in 1.5 dB steps.
   left_adc_gain = sound->left.adc_gain; right_adc_gain = sound->right.adc_gain;
// Set channel.mic levels, 1bit->2bit, non-zero enables 20 dB MIC input gain.
   left_mic_gain_enable  = sound->left.mic_gain_enable; right_mic_gain_enable = sound->right.mic_gain_enable;
// AUX-1 Settings - 0-31 aux1 mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes aux1 mix.
   left_aux1_mix_gain  = sound->left.aux1_mix_gain;  left_aux1_mix_mute  = sound->left.aux1_mix_mute;
   right_aux1_mix_gain = sound->right.aux1_mix_gain; right_aux1_mix_mute = sound->right.aux1_mix_mute;
// AUX-2 Settings - 0-31 aux2 mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes aux2 mix.
   left_aux2_mix_gain  = sound->left.aux2_mix_gain;  left_aux2_mix_mute  = sound->left.aux2_mix_mute;
   right_aux2_mix_gain = sound->right.aux2_mix_gain; right_aux2_mix_mute = sound->right.aux2_mix_mute;
// Line Levels Settings - 0-31 line mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes line mix.
   left_line_mix_gain  = sound->left.line_mix_gain;  left_line_mix_mute  = sound->left.line_mix_mute;
   right_line_mix_gain = sound->right.line_mix_gain; right_line_mix_mute = sound->right.line_mix_mute;
// DAC Settings - 0..61 dac attenuation, in -1.5 dB steps. Non-zero mutes dac output.
   left_dac_attn  = sound->left.dac_attn;  left_dac_mute  = sound->left.dac_mute;
   right_dac_attn = sound->right.dac_attn; right_dac_mute = sound->right.dac_mute;
// Only support 48 kHz. Sample rate from enum table.
   sound_sample_rate = sound->sample_rate;
// Only support 8 bit mono playback, sample format for playback.
   sound_playback_format = sound->playback_format;
// Only support 8 bit mono capture, sample format for capture.
   sound_capture_format = sound->capture_format;
// Control for dither_enable, non-zero enables dither on 16 => 8 bit.
   sound_dither_enable = sound->dither_enable;
// Control for loop_attn. 0..64 adc to dac loopback attenuation, in -1.5 dB steps. Non-zero enables loopback.
   sound_loop_attn = sound->loop_attn; sound_loop_enable = sound->loop_enable;
// Control for output boost, zero (2.0 Vpp) non-zero (2.8 Vpp) output level boost.
   sound_output_boost = sound->output_boost;
// Control for high pass filter, non-zero enables highpass filter in adc.
   sound_highpass_enable = sound->highpass_enable;
// Control for speaker gain, 0..64 mono speaker gain. Un/Mute speaker, non-zero mutes speaker.
   sound_mono_gain = sound->mono_gain; sound_mono_mute = sound->mono_mute;
   if (debug) {
      dprintf("PPA: ADC Sources (enum) => %d %d \n", (int)left_adc_source, (int)right_adc_source);
      dprintf("PPA: ADC Gains => %d %d \n",   (int)left_adc_gain, (int)right_adc_gain);
      dprintf("PPA: Mic Enables => %d %d \n", (int)left_mic_gain_enable, (int)right_mic_gain_enable);
      dprintf("PPA: Left  Aux1 Gain-Mute => %d %d \n", (int)left_aux1_mix_gain,  (int)left_aux1_mix_mute);
      dprintf("PPA: Right Aux1 Gain-Mute => %d %d \n", (int)right_aux1_mix_gain, (int)right_aux1_mix_mute);
      dprintf("PPA: Left  Aux2 Gain-Mute => %d %d \n", (int)left_aux2_mix_gain,  (int)left_aux2_mix_mute);
      dprintf("PPA: Right Aux2 Gain-Mute => %d %d \n", (int)right_aux2_mix_gain, (int)right_aux2_mix_mute);
      dprintf("PPA: Left  Line Gain-Mute => %d %d \n", (int)left_line_mix_gain,  (int)left_line_mix_mute);
      dprintf("PPA: Right Line Gain-Mute => %d %d \n", (int)right_line_mix_gain, (int)right_line_mix_mute);
      dprintf("PPA: Left  DAC Mute-Attenuation => %d %d \n", (int)left_dac_mute,  (int)left_dac_attn);
      dprintf("PPA: Right DAC Mute-Attenuation => %d %d \n", (int)right_dac_mute, (int)right_dac_attn);
      dprintf("PPA: sound_sample_rate (enum) => %d \n", (int)sound_sample_rate);
      dprintf("PPA: sound_playback_format (enum) => %d \n", (int)sound_playback_format);
      dprintf("PPA: sound_capture_format (enum) => %d \n", (int)sound_capture_format);
      dprintf("PPA: sound->dither_enable (enum) => %d \n", (int)sound_dither_enable);
      dprintf("PPA: sound_loop_attn, sound_loop_enable => %d %d \n", (int)sound_loop_attn, (int)sound_loop_enable);
      dprintf("PPA: sound->output_boost => %d \n", (int)sound_output_boost);
      dprintf("PPA: sound_highpass_enable => %d \n", (int)sound_highpass_enable);
      dprintf("PPA: Sound Mono Gain => %d  %d \n", (int)sound_mono_gain, (int)sound_mono_mute);
      dprintf("\n"); }; // Prints a gap for readability.
   return B_OK; }
/*--------------------------------------------------------------------------------------------------------------*/
status_t get_parameters(sound_setup *sound) {
   if (debug) dprintf("PPA: get_parameters.\n");
// Mixer sources to digitalize.
   sound->left.adc_source = left_adc_source; sound->right.adc_source = right_adc_source;
// Controls for channel.adc_gain. 0..15 adc gain, in 1.5 dB steps.
   sound->left.adc_gain = left_adc_gain; sound->right.adc_gain = right_adc_gain;
// Set channel.mic levels, 1bit->2bit, non-zero enables 20 dB MIC input gain.
   sound->left.mic_gain_enable  = left_mic_gain_enable; sound->right.mic_gain_enable = right_mic_gain_enable;
// AUX-1 Settings - 0-31 aux1 mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes aux1 mix.
   sound->left.aux1_mix_gain  = left_aux1_mix_gain;  sound->left.aux1_mix_mute  = left_aux1_mix_mute;
   sound->right.aux1_mix_gain = right_aux1_mix_gain; sound->right.aux1_mix_mute = right_aux1_mix_mute;
// AUX-2 Settings - 0-31 aux2 mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes aux2 mix.
   sound->left.aux2_mix_gain  = left_aux2_mix_gain;  sound->left.aux2_mix_mute  = left_aux2_mix_mute;
   sound->right.aux2_mix_gain = right_aux2_mix_gain; sound->right.aux2_mix_mute = right_aux2_mix_mute;
// Line Levels Settings - 0-31 line mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes line mix.
   sound->left.line_mix_gain  = left_line_mix_gain;  sound->left.line_mix_mute  = left_line_mix_mute;
   sound->right.line_mix_gain = right_line_mix_gain; sound->right.line_mix_mute = right_line_mix_mute;
// DAC Settings - 0..61 dac attenuation, in -1.5 dB steps. Non-zero mutes dac output.
   sound->left.dac_attn  = left_dac_attn;  sound->left.dac_mute  = left_dac_mute;
   sound->right.dac_attn = right_dac_attn; sound->right.dac_mute = right_dac_mute;
// Only support 48 kHz. Sample rate from enum table.
   sound->sample_rate = sound_sample_rate;
// Only support 8 bit mono playback, sample format for playback.
   sound->playback_format = sound_playback_format;
// Only support 8 bit mono capture, sample format for capture.
   sound->capture_format = sound_capture_format;
// Control for dither_enable, non-zero enables dither on 16 => 8 bit.
   sound->dither_enable = sound->dither_enable;
// Control for loop_attn. 0..64 adc to dac loopback attenuation, in -1.5 dB steps. Non-zero enables loopback.
   sound->loop_attn = sound_loop_attn; sound->loop_enable = sound_loop_enable;
// Control for output boost, zero (2.0 Vpp) non-zero (2.8 Vpp) output level boost.
   sound->output_boost = sound_output_boost;
// Control for high pass filter, non-zero enables highpass filter in adc.
   sound->highpass_enable = sound_highpass_enable;
// Control for speaker gain, 0..64 mono speaker gain.
   sound->mono_gain = sound_mono_gain;
// Un/Mute speaker, non-zero mutes speaker.
   sound->mono_mute = sound_mono_mute;
   if (debug) {
      dprintf("PPA: ADC Sources (enum) => %d %d \n", (int)left_adc_source, (int)right_adc_source);
      dprintf("PPA: ADC Gains => %d %d \n",   (int)left_adc_gain, (int)right_adc_gain);
      dprintf("PPA: Mic Enables => %d %d \n", (int)left_mic_gain_enable, (int)right_mic_gain_enable);
      dprintf("PPA: Left  Aux1 Gain-Mute => %d %d \n", (int)left_aux1_mix_gain,  (int)left_aux1_mix_mute);
      dprintf("PPA: Right Aux1 Gain-Mute => %d %d \n", (int)right_aux1_mix_gain, (int)right_aux1_mix_mute);
      dprintf("PPA: Left  Aux2 Gain-Mute => %d %d \n", (int)left_aux2_mix_gain,  (int)left_aux2_mix_mute);
      dprintf("PPA: Right Aux2 Gain-Mute => %d %d \n", (int)right_aux2_mix_gain, (int)right_aux2_mix_mute);
      dprintf("PPA: Left  Line Gain-Mute => %d %d \n", (int)left_line_mix_gain,  (int)left_line_mix_mute);
      dprintf("PPA: Right Line Gain-Mute => %d %d \n", (int)right_line_mix_gain, (int)right_line_mix_mute);
      dprintf("PPA: Left  DAC Mute-Attenuation => %d %d \n", (int)left_dac_mute,  (int)left_dac_attn);
      dprintf("PPA: Right DAC Mute-Attenuation => %d %d \n", (int)right_dac_mute, (int)right_dac_attn);
      dprintf("PPA: sound_sample_rate (enum) => %d \n", (int)sound_sample_rate);
      dprintf("PPA: sound_playback_format (enum) => %d \n", (int)sound_playback_format);
      dprintf("PPA: sound_capture_format (enum) => %d \n", (int)sound_capture_format);
      dprintf("PPA: sound->dither_enable (enum) => %d \n", (int)sound_dither_enable);
      dprintf("PPA: sound_loop_attn, sound_loop_enable => %d %d \n", (int)sound_loop_attn, (int)sound_loop_enable);
      dprintf("PPA: sound->output_boost => %d \n", (int)sound_output_boost);
      dprintf("PPA: sound_highpass_enable => %d \n", (int)sound_highpass_enable);
      dprintf("PPA: Sound Mono Gain => %d  %d \n\n", (int)sound_mono_gain, (int)sound_mono_mute);
      dprintf("\n"); }; // Prints a gap for readability.
   return B_OK; }

/*****************************************************************************************************************
 read_hook() - status_t read_hook(void *cookie, off_t position, void *data, size_t *length) - This hook handles 
 the Posix read() function for an open instance of your driver.  Implement it to read len bytes of data starting
 at the specified byte position on the device, storing the read bytes at data. Exactly what this does is
 device-specific (disk devices would read from the specified offset on the disk, but a graphics driver might have
 some other interpretation of this request). Before returning, you should set len to the actual number of bytes
 read into the buffer. Return B_OK if data was read (even if the number of returned bytes is less than requested),
 otherwise return an appropriate error.
 
 Sound Driver notes: This code does not work properly.  I don't know why yet! */

static status_t device_read(void *cookie, off_t pos, void *data_buffer, size_t *length) {
 int index; long record_buffer_length;

 acquire_sem(read_lock_sem); // Block extra reads from buffer before it is loaded into system.
 index = 0; record_buffer_length = *length; // Setup to copy to data buffer.
 memcpy(data_buffer, record_buffer, record_buffer_length); // Copy fake sound data to play buffer.
                                                           // Filled read buffer with a test (sawtooth) wave form.
 release_sem_etc(read_done_sem, 1, B_DO_NOT_RESCHEDULE);   // Release semaphore after data copied to databuffer.
 release_sem(read_lock_sem); return B_OK; }                   // Allow more reads.


/*****************************************************************************************************************
 write_hook() - status_t write_hook(void *cookie, off_t position, void *data, size_t length) -  This hook handles
 the Posix write() function for an open instance of your driver. Implement it to write len bytes of data starting
 at the specified byte position on the device, from the buffer pointed to by data. Exactly what this does is
 device-specific (disk devices would write to the specified offset on the disk, but a graphics driver might have
 some other interpretation of this request). Return B_OK if data was read (even if the number of returned bytes is
 less than requested), otherwise return an appropriate error.

 Sound Driver notes: This driver assumes the buffer is filled with stereo signed sixteen(16) bit sound samples,
 it outputs the samples thru the parallel (printer) port as a eight(8) mono audio stream which are feed into an
 eight(8) Bit DAC attached to the port.  Notice we transfer the raw data to the driver's own memory as soon as
 possible.  The data pointed to in the write_hook() can not be considered stable. Why?  I don't know.
 
 Warning!!! PLAY_CHUNK should be small enough to be processed completely between two sound samples, ie
 frame_length in microseconds.  Otherwise there may be timing problems. Special note: Since changing the code to
 run with a ring buffer, interrupt modes can now output data independant of input.  It should be okay to increase
 PLAY_CHUNK.  Experiment if you wish. */

static status_t device_write(void *cookie, off_t pos, const void *data_buffer, size_t *length) {
 int index, interpolation; long data_length, convert, play_end;
 static uchar old_sample;  // Contains previous audio sample needed for Interpolation.
 static uchar new_sample;  // Contains latest audio sample needed for Interpolation.
 #define PLAY_LENGTH 20000 // Size of temp buffer.
 #define PLAY_CHUNK 256    // Amount of sound data to process at a single time.
 static uint16 play_buffer[PLAY_LENGTH + 4]; /* Buffer for 16 bit stereo data of sound to be processed.  An area
 to copy the data pointed to by the write_device(*buffer) because for some reason the original data pointed to by
 write_device(*buffer) is not considered stable.  Best guess, BeOS uses two or more sound buffer areas that it
 re-uses in cyclic fashion.  While the driver is playing one buffer the next is being filled, but if the driver is
 slow then the first buffer will be caught up to and have data starting to be written into it while is contents
 are being played.  Not good. */
 static long play_length;   // Number of elements in to transfer to ring buffer.
 static long output_rate;   // Samples per second from ring_buffer to output port.
 static long sample_length; // How many (integer) microseconds between device_write(*data_buffer) samples.

 if (debug) dprintf("PPA: Write to Driver.\n\n"); 
 acquire_sem(write_lock_sem); // Block any more device_write() before data_buffer is transfered to play_buffer.

 interpolation = interpolate;   // Copies state so it can not change value in the middle of code.
 timing = timing_mode;          // Copies state so it can not change value in the middle of code.

 sampling_rate = (long)(AudioFormat.sample_rate); // Samples per second of data in play_buffer.
 output_rate = sampling_rate * interpolation;     // Samples per second from ring_buffer to output port.

 sample_length = 1000000L / sampling_rate; // Microseconds (integer) between device_write(*data_buffer) samples.
 frame_length  = 1000000L / output_rate;   // Microseconds (integer) between output samples.
 frame_rate    = 1000000L / frame_length;  // The sampling rate the OS's interrupts allows.

 if (old_timing > 4) {                           // Test if previous buffer was played with an interrupt handler.
    if ((old_timing != timing)|(old_frame_length != frame_length)) {
       while (read_index !=write_index) { };     // Wait till ring_buffer is empty.
       cancel_timer((timer *)&device_timer); } } // Yes, cancel old timer().    
/*--------------------------------------------------------------------------------------------------------------*/
 data_length = (long)(*length);                                 // Setup length to copy from data_buffer.
 if (data_length > PLAY_LENGTH) {                               // Test if more data than play_buffer can hold.
    data_length = PLAY_LENGTH; *length = (size_t)data_length; } // Amount copied to play_buffer.
 memcpy(play_buffer, data_buffer, data_length);                 // Copy sound data in data_buffer to play_buffer.
 play_length = data_length / 2;                   // Number of 16 bit samples in play_buffer. Note stereo sound.
 /*--------------------------------------------------------------------------------------------------------------*/
 next_frame_time = (1 + (system_time()/sample_length)) * sample_length; // Quantumize the timing so that each
                              // audio sample is outputted at regular timing even between calls to device_write().
 next_buffer_time = next_frame_time + (bigtime_t)(play_length / 2 * frame_length * interpolation); // End of present buffer.


 index = 0; play_end = 0;       // Point indexs to start of play_buffer.
 while (index < play_length) {  // And data left in play_buffer?
       play_end += PLAY_CHUNK;  // Next block of data to play.
       if (play_end > play_length) {
          play_end = play_length; } // Don't run pass end of buffer.

       if (interpolation == 1) {
          for ( ; index < play_end; index+=2 ) {                     // Create 8 bit mono version of sound data.
              ring_buffer[write_index++] = (uchar)((play_buffer[index] >> 8) + 128); } // INT16 = UINT8.
          old_sample = ring_buffer[write_index - 1]; }               // Copy last sample written to ring_buffer.
/*--------------------------------------------------------------------------------------------------------------*/
       if (interpolation == 2) {
          for (; index < play_end; index+=2 ) {                      // Create 8 bit version of sound data.
              new_sample = (uchar)((play_buffer[index] >> 8) + 128); // Convert signed 16 Bit to unsigned 8 Bit.
              convert = ((long)old_sample << 8) + (long)new_sample;  // old_sample * 256 + new_sample.
              ring_buffer[write_index++] = c50[convert];             // (Old+New)/2
              ring_buffer[write_index++] = new_sample;               // (Old*0+New*1)/1
          old_sample = new_sample; } }                               // Copy last sample written to ring_buffer.
/*--------------------------------------------------------------------------------------------------------------*/
       if (interpolation == 3) {
          for (; index < play_end; index+=2 ) {                      // Create 8 bit version of sound data.
              new_sample = (uchar)((play_buffer[index] >> 8) + 128); // Convert signed 16 Bit to unsigned 8 Bit.
              convert = ((long)old_sample << 8) + (long)new_sample;  // old_sample * 256 + new_sample.
              ring_buffer[write_index++] = c33[convert];             // (Old*2+New*1)/3
              ring_buffer[write_index++] = c66[convert];             // (Old*1+New*2)/3
              ring_buffer[write_index++] = new_sample;               // (Old*0+New*1)/1
          old_sample = new_sample; } }                               // Copy last sample written to ring_buffer.
/*--------------------------------------------------------------------------------------------------------------*/
       if (interpolation == 4) {
          for (; index < play_end; index++ ) {                       // Create 8 bit version of sound data.
              new_sample = (uchar)((play_buffer[index] >> 8) + 128); // Convert signed 16 Bit to unsigned 8 Bit.
              convert = ((uint16)old_sample<<8)+(uint16)new_sample;  // old_sample * 256 + new_sample.
              ring_buffer[write_index++] = c25[convert];             // (Old*3+New*1)/4
              ring_buffer[write_index++] = c50[convert];             // (Old*2+New*2)/4
              ring_buffer[write_index++] = c75[convert];             // (Old*1+New*3)/4
              ring_buffer[write_index++] = new_sample;               // (Old*0+New*4)/4
          old_sample = new_sample; } }                               // Copy last sample written to ring_buffer.

/* Sound Driver notes: These are the diffirent methods I developed to time and control the outputting of the audio
 data written to the parallel port.  Notice when I use the busy loop "While (System_Time() < Next_Frame_Time) { }"
 that I also wrap it with the option to turn off interrupts to improve the timing.
 
 Also you will notice that except for 'play_chunk_mode_2()' that I try to avoid sending data to the parallel port
 if that value is already there.  Sending the data every time will make no diffirence as 'play_chunk_mode_2' will
 prove, but writes to an I/O port are expensive in CPU TIME.  In my tests I can not push more than 2.5 million
 writes per second to the parallel port no matter what my CPU speed is set at.  This suggests each I/O transaction
 take 400 nano-seconds to perform, that is a big hit in this day of 2+Ghz machines. 
 
 Timing of the bytes sent to the parallel (printer) port for the DAC can be by one of six(6) methods.
 (1) Busy Wait Loop, (2) Basic Snooze_Until(), (3) Snooze_Until() & skip duplicate output values.
 (4) Snooze_Until(-Latency) & Interrupt Disable & Busy Wait Loop & skip duplicate output values.
 (5) B_PERIODIC_TIMER. (5) Fixed B_ONE_SHOT_ABSOLUTE_TIMER (6) Variable B_ONE_SHOT_ABSOLUTE_TIMER. */
       if (timing == 1) {   // Time output with a Busy-Wait-Loop. (Historic reasons only).
          play_chunk_mode_1(); } // Output ring_buffer sound samples using timing mode 1.
       if (timing == 2) {   // Use only Snooze_Until() to control output timing.
          play_chunk_mode_2(); } // Output ring_buffer sound samples using timing mode 2.
       if (timing == 3) {   // Use Snooze_Until() & skip duplicate sample values.
          play_chunk_mode_3(); } // Output ring_buffer sound samples using timing mode 3.
       if (timing == 4) {   // Snooze_Until(-Latency), Interrupt Disable, Busy Wait Loop & skip repeats.
          play_chunk_mode_4(); } // Output ring_buffer sound samples using timing mode 4.
       if (timing == 5) {   // Interrupt B_PERIODIC_TIMER.
          play_chunk_mode_5( (bigtime_t) frame_length); }    // Output ring_buffer sound samples using timing mode 5.
 } // End of While
 
 release_sem_etc(write_done_sem, 1, B_DO_NOT_RESCHEDULE);       // Release semaphore after sound data copied.
 release_sem(write_lock_sem); return B_OK; } // Allow new calls to device_write.

/*****************************************************************************************************************
 This is the most basic of timing code.  Called a Busy Wait Loop it has a very high CPU loading.  I only included
 here because it is easy to write and understand, and it has some historical interest. */

static status_t play_chunk_mode_1() {
static uchar old_ring_output, new_ring_output; cpu_status st;

 while (read_index !=write_index) {                 // Test ring_buffer does contain more sound samples.
       new_ring_output = ring_buffer[read_index++]; // Get next sample to play.

       if (old_ring_output != new_ring_output) {      // Test if change in output signal value.
          if (interrupt_flag == true) {               // Test if system interrupts are to be disabled.
             st = disable_interrupts(); }             // If yes, disable interrupts to improve timing.
          while (system_time() < next_frame_time) { } // Busy-Wait, loop until time to play next sample.
          isa->write_io_8(0x378, new_ring_output);    // If yes.  Write new signal data to parallel port.
          if (interrupt_flag == true) {               // Test if system interrupts were disabled.
             restore_interrupts(st); }                // Enable interrupts to maintain OS requirements.
          old_ring_output = new_ring_output; }        // Update old_ring_output to match present output.

       next_frame_time += frame_length; } // Time for next sound sample out.

 while (system_time() < (next_frame_time - frame_length)) { } // Matchs the output timing to the ring_buffer's if
                                              // two or more of the last values in the ring_buffer are the same.
 old_timing = timing; old_frame_length = frame_length; return; }

/*****************************************************************************************************************
 This shows how the Snooze_Until() function is used to control timing of sound samples.  This is code is the most
 basic design with a far lower CPU LOAD than use of a Busy Wait Loop. But since it always write each and every
 ring_buffer value to the parallel port it does have a higher CPU LOAD than is really needed for the job. */

static status_t play_chunk_mode_2() {
static uchar old_ring_output, new_ring_output; cpu_status st;

 while (read_index !=write_index) {                 // Test ring_buffer does contain more sound samples.
       new_ring_output = ring_buffer[read_index++]; // Get next sample to play.

       snooze_until(next_frame_time, B_SYSTEM_TIMEBASE); // Sleep until time to play sound sample.
       isa->write_io_8(0x378, new_ring_output);          // Write new signal data to parallel port.
       old_ring_output = new_ring_output;                // Update old_ring_output to match present output.

       next_frame_time += frame_length; } // Time for next sound sample out.

 old_timing = timing; old_frame_length = frame_length; return B_OK; }

/*****************************************************************************************************************
 Again, uses the Snooze_Until() function to control timing of sound samples.  But now if the value to be outputted
 has not change the I/O call is skipped.  An interesting and useful side effect of this code is that the timing of
 how long the thread should sleep till the next valid output is done is still inside this loop, so if we have say
 five same output values in a row the Snooze_Until() gets called only once they have all been skipped over and the
 time the thread is told to snooze to is the same as the time stamp of the changing output value. */

static status_t play_chunk_mode_3() {
static uchar old_ring_output, new_ring_output; cpu_status st;

 while (read_index !=write_index) {                 // Test ring_buffer does contain more sound samples.
       new_ring_output = ring_buffer[read_index++]; // Get next sample to play.

       if (old_ring_output != new_ring_output) {            // Test if change in output signal value.
          snooze_until(next_frame_time, B_SYSTEM_TIMEBASE); // Yes, sleep until time to play sound sample.
          isa->write_io_8(0x378, new_ring_output);          // Write new signal data to parallel port.
          old_ring_output = new_ring_output; }              // Update old_ring_output to match present output.

       next_frame_time += frame_length; } // Time for next sound sample out.

 snooze_until(next_frame_time - frame_length, B_SYSTEM_TIMEBASE); // Matchs the output timing to the ring_buffer's
                                             // if two or more of the last values in the ring_buffer are the same.
 old_timing = timing; old_frame_length = frame_length; return B_OK; }

/*****************************************************************************************************************
 This is the expanded use of the Snooze_Until() function.  Assuming that you already understand the code in
 'play_chunk_mode_3()' the following changes were made.  First, Snooze_Until() can not be consider a very accurate
 timer function as other interrupts could interfer with it's operation.  Latency is a varible that allows us to
 tell the driver to end the snooze a little early, at which point we enter a tight Busy Wait Loop that gives us
 some fine timing control.  Because interrupts can even interfer with timing from 'system_time()' we have the
 option to turn off interrupts during the Busy Wait Loop to improve the timing even more. */

static status_t play_chunk_mode_4() {
static uchar old_ring_output, new_ring_output; cpu_status st;

 while (read_index !=write_index) {                 // Test ring_buffer does contain more sound samples.
       new_ring_output = ring_buffer[read_index++]; // Get next sample to play.

       if (old_ring_output != new_ring_output) {         // Test if change in output signal value.
          snooze_until(next_frame_time - Latency, B_SYSTEM_TIMEBASE); // Sleep until time to play sound sample.

          if (Latency > 0) {                               // Test for Latency.
             if (interrupt_flag == true) {                 // Test if system interrupts are to be disabled.
                st = disable_interrupts(); }               // If yes, disable interrupts to improve timing.
             while (system_time() < next_frame_time) { } } // Busy-Wait, loop until time to play next sample.

          isa->write_io_8(0x378, new_ring_output);         // Write new signal data to parallel port.

          if (Latency > 0) {                               // Test for Latency.
             if (interrupt_flag == true) {                 // Test if system interrupts were disabled.
                restore_interrupts(st); } }                // Enable interrupts to maintain OS requirements.

          old_ring_output = new_ring_output; }             // Update old_ring_output to match present output.

       next_frame_time += frame_length; } // Time for next sound sample out.

 snooze_until(next_frame_time - frame_length, B_SYSTEM_TIMEBASE); // Matchs the output timing to the ring_buffer's
                                             // if two or more of the last values in the ring_buffer are the same.
 old_timing = timing; old_frame_length = frame_length; return B_OK; }

/****************************************************************************************************************/

static status_t play_chunk_mode_5( bigtime_t frame_length) {

 if (old_timing == 5) {                      // Previous Timer() was B_PERIODIC_TIMER.
    if (old_frame_length == frame_length) {  // Timer was using same frame_length as present frame_length.
       return B_HANDLED_INTERRUPT; } }       // No changes needed.  Exit.
    
 add_timer((timer*)&device_timer,         // If no, add an interrupt handler.
     (timer_hook)device_simple_interrupt, // Where the interrupt will execute.
     (bigtime_t)frame_length,             // Time between ring_buffer samples.
     (int32) B_PERIODIC_TIMER);           // Interrupt will automaticly repeat every frame_length microseconds.

 old_timing = timing; old_frame_length = frame_length; return B_HANDLED_INTERRUPT; }

/*--------------------------------------------------------------------------------------------------------------*/
static int32 device_simple_interrupt(timer *_timing, uint32 flags) {
static uchar old_ring_output, new_ring_output;

 if (read_index !=write_index) {                 // Test ring_buffer does contain more sound samples.
    new_ring_output = ring_buffer[read_index++]; // Get next sample to play.

    if (old_ring_output != new_ring_output) {   // Test if change in output signal value.
       isa->write_io_8(0x378, new_ring_output); // If yes.  Write new signal data to parallel port.
       old_ring_output = new_ring_output; } }   // Update old_ring_output to match present output.

 return B_HANDLED_INTERRUPT; }

/*****************************************************************************************************************
 publish_devices - const char** publish_devices(void) - returns a null-terminated array of devices supported by
 this driver.  Devfs calls publish_devices() to learn the names, relative to /dev, of the devices the driver
 supports. The driver should return a NULL-terminated array of strings indicating all the installed devices the
 driver supports. For example, an ethernet device driver might return: 
 static char *devices[] = { "net/ether", NULL };

 In this case, devfs will then create the pseudo-file /dev/net/ether, through which all user applications can
 access the driver.  Since only one instance of the driver will be loaded, if support for multiple devices of the
 same type is desired, the driver must be capable of supporting them. If the driver senses (and supports) two
 ethernet cards, it might return: 
 static char *devices[] = { "net/ether1", "net/ether2", NULL };

 Sound Driver notes: This device appears at devs/audio/old/sound. */

const char** publish_devices(void) {
 return driver_names; }

/*****************************************************************************************************************
 find_device - device_hooks* find_device(const char *name) - returns a pointer to device hooks structure for a
 given device name.  When a device published by the driver is accessed, devfs communicates with it through a
 series of hook functions that handle the requests.  The find_device() function is called to obtain a list of
 these hook functions, so that devfs can call them.  The device_hooks structure returned lists out the hook
 functions.  The device_hooks structure, and what each hook does, is described in the next section. */

device_hooks *find_device(const char *name) {
 int i; // dprintf("PPA: Look at device names.\n"); 
 for (i=0; driver_names[i]; i++) {
     if (!strncmp(name, driver_names[i], B_OS_NAME_LENGTH)) {
        // dprintf("PPA: Found matching %s driver.\n\n",driver_names[i]);
        return &sound_device[i]; } }
 // dprintf("PPA: No driver match found.\n\n");
 return NULL; }

/*****************************************************************************************************************
 status_t readv_hook(void *cookie, off_t position, const struct iovec *vec, size_t count, size_t *length)
 This hook handles the Posix readv() function for an open instance of your driver.  This is a scatter/gather read
 function; given an array of iovec structures describing address/length pairs for a group of destination buffers,
 your implementation should fill each successive buffer with bytes, up to a total of len bytes. The vec array has
 count items in it. As with read_hook(), set len to the actual number of bytes read, and return an appropriate
 result code.

static status_t my_device_readv(void *cookie, off_t position, const iovec *vec, size_t count, size_t *length) {
 // dprintf("PPA: Readv driver.\n");
 return B_OK; }

/*****************************************************************************************************************
 status_t writev_hook(void *cookie, off_t position, const struct iovec *vec, size_t count, size_t *length) 
 This hook handles the Posix writev() function for an open instance of your driver. This is a scatter/gather write
 function; given an array of iovec structures describing address/length pairs for a group of source buffers, your
 implementation should write each successive buffer to disk, up to a total of len bytes. The vec array has count
 items in it. Before returning, set len to the actual number of bytes written, and return an appropriate result
 code.

static status_t my_device_writev(void *cookie, off_t position, const iovec *vec, size_t count, size_t *length) {
 // dprintf("PPA: Writev driver.\n");
 return B_OK; }

/*****************************************************************************************************************
select_hook() , deselect_hook() 
 These hooks are reserved for future use. Set the corresponding entries in your device_hooks structure to NULL.

status_t my_device_select(void *cookie, uint8 event, uint32 ref, selectsync *sync) {
 // dprintf("PPA: Select driver.\n");
 return B_OK; }

status_t my_device_deselect(void *cookie, uint8 event, selectsync *sync) {
 // dprintf("PPA: Deselect driver.\n");
 return B_OK; } */
